// Filename: encryptStream.h
// Created by:  drose (01Sep04)
//
////////////////////////////////////////////////////////////////////
//
// PANDA 3D SOFTWARE
// Copyright (c) Carnegie Mellon University.  All rights reserved.
//
// All use of this software is subject to the terms of the revised BSD
// license.  You should have received a copy of this license along
// with this source code in a file named "LICENSE."
//
////////////////////////////////////////////////////////////////////

#ifndef ENCRYPTSTREAM_H
#define ENCRYPTSTREAM_H

#include "dtoolbase.h"

// This module is not compiled if OpenSSL is not available.
#ifdef HAVE_OPENSSL

#include "encryptStreamBuf.h"

////////////////////////////////////////////////////////////////////
//       Class : IDecryptStream
// Description : An input stream object that uses OpenSSL to decrypt
//               the input from another source stream on-the-fly.
//
//               Attach an IDecryptStream to an existing istream that
//               provides encrypted data, as generated by an
//               OEncryptStream, and read the corresponding
//               unencrypted data from the IDecryptStream.
//
//               Seeking is not supported.
////////////////////////////////////////////////////////////////////
class EXPCL_DTOOLCONFIG IDecryptStream : public istream {
PUBLISHED:
  INLINE IDecryptStream();
  INLINE IDecryptStream(istream *source, bool owns_source,
                        const string &password);

  INLINE IDecryptStream &open(istream *source, bool owns_source,
                              const string &password);
  INLINE IDecryptStream &close();

  INLINE const string &get_algorithm() const;
  INLINE int get_key_length() const;
  INLINE int get_iteration_count() const;

private:
  EncryptStreamBuf _buf;
};

////////////////////////////////////////////////////////////////////
//       Class : OEncryptStream
// Description : An input stream object that uses OpenSSL to encrypt
//               data to another destination stream on-the-fly.
//
//               Attach an OEncryptStream to an existing ostream that
//               will accept encrypted data, and write your
//               unencrypted source data to the OEncryptStream.
//
//               Seeking is not supported.
////////////////////////////////////////////////////////////////////
class EXPCL_DTOOLCONFIG OEncryptStream : public ostream {
PUBLISHED:
  INLINE OEncryptStream();
  INLINE OEncryptStream(ostream *dest, bool owns_dest, 
                        const string &password);

  INLINE OEncryptStream &open(ostream *dest, bool owns_dest, 
                              const string &password);
  INLINE OEncryptStream &close();

  INLINE void set_algorithm(const string &algorithm);
  INLINE void set_key_length(int key_length);
  INLINE void set_iteration_count(int iteration_count);

private:
  EncryptStreamBuf _buf;
};

#include "encryptStream.I"

#endif  // HAVE_OPENSSL


#endif


