// Filename: projectionScreen.I
// Created by:  drose (11Dec01)
//
////////////////////////////////////////////////////////////////////
//
// PANDA 3D SOFTWARE
// Copyright (c) 2001 - 2004, Disney Enterprises, Inc.  All rights reserved
//
// All use of this software is subject to the terms of the Panda 3d
// Software license.  You should have received a copy of this license
// along with this source code; you will also find a current copy of
// the license at http://etc.cmu.edu/panda3d/docs/license/ .
//
// To contact the maintainers of this program write to
// panda3d-general@lists.sourceforge.net .
//
////////////////////////////////////////////////////////////////////



////////////////////////////////////////////////////////////////////
//     Function: ProjectionScreen::get_projector
//       Access: Published
//  Description: Returns the NodePath to the LensNode that is to serve
//               as the projector for this screen, or empty if no
//               projector is associated.
////////////////////////////////////////////////////////////////////
INLINE const NodePath &ProjectionScreen::
get_projector() const {
  return _projector;
}

////////////////////////////////////////////////////////////////////
//     Function: ProjectionScreen::set_texcoord_name
//       Access: Published
//  Description: Specifies the name of the texture coordinates that
//               are generated by this particular ProjectionScreen.
//               This can be used in the presence of multitexturing to
//               compute the UV's for just a subset of all of the
//               active stages of the multitexture pipeline.
////////////////////////////////////////////////////////////////////
INLINE void ProjectionScreen::
set_texcoord_name(const string &texcoord_name) {
  _texcoord_name = TexCoordName::make(texcoord_name);
  _stale = true;
}

////////////////////////////////////////////////////////////////////
//     Function: ProjectionScreen::get_texcoord_name
//       Access: Published
//  Description: Returns the name of the texture coordinates that
//               will be generated by this particular
//               ProjectionScreen, as set by set_texcoord_name().
////////////////////////////////////////////////////////////////////
INLINE const string &ProjectionScreen::
get_texcoord_name() const {
  return _texcoord_name->get_name();
}

////////////////////////////////////////////////////////////////////
//     Function: ProjectionScreen::set_invert_uvs
//       Access: Published
//  Description: Some OpenGL graphics drivers are known to invert the
//               framebuffer image when they copy it to texture.
//               (This is arguably a problem with the OpenGL spec,
//               which seems to be unclear about the proper ordering
//               of pixels in this operation.)
//
//               In any case, set this true to compensate for this
//               effect by inverting the UV's of the projection
//               screen.  The default is taken from the Configrc
//               variable project-invert-uvs.
////////////////////////////////////////////////////////////////////
INLINE void ProjectionScreen::
set_invert_uvs(bool invert_uvs) {
  _invert_uvs = invert_uvs;
  _stale = true;
}


////////////////////////////////////////////////////////////////////
//     Function: ProjectionScreen::get_invert_uvs
//       Access: Published
//  Description: Returns whether this screen is compensating for a
//               graphics driver inverting the framebuffer image.  See
//               set_invert_uvs().
////////////////////////////////////////////////////////////////////
INLINE bool ProjectionScreen::
get_invert_uvs() const {
  return _invert_uvs;
}

////////////////////////////////////////////////////////////////////
//     Function: ProjectionScreen::set_vignette_on
//       Access: Published
//  Description: Specifies whether vertex-based vignetting should be
//               on.  When this is enabled, vertex color will be set
//               on the screen vertices to color the screen two
//               distinct colors, usually white and black, for the
//               parts of the screen in front of and outside the lens'
//               frustum, respectively.  When this is not enabled, the
//               screen color will be left alone.
//
//               This effect generally looks terrible, but it does
//               at least make the boundaries of the lens clear.
////////////////////////////////////////////////////////////////////
INLINE void ProjectionScreen::
set_vignette_on(bool vignette_on) {
  _vignette_on = vignette_on;
  _stale = true;
}


////////////////////////////////////////////////////////////////////
//     Function: ProjectionScreen::get_vignette_on
//       Access: Published
//  Description: Returns true if vertex-based vignetting is on, false
//               otherwise.  See set_vignette_on().
////////////////////////////////////////////////////////////////////
INLINE bool ProjectionScreen::
get_vignette_on() const {
  return _vignette_on;
}

////////////////////////////////////////////////////////////////////
//     Function: ProjectionScreen::set_vignette_color
//       Access: Published
//  Description: Specifies the color the screen will be painted at the
//               portions outside of the lens' frustum; i.e. where the
//               lens can't see it or illuminate it.  This color is
//               only used if the vignette_on flag is true; see
//               set_vignette_on().
////////////////////////////////////////////////////////////////////
INLINE void ProjectionScreen::
set_vignette_color(const Colorf &vignette_color) {
  _vignette_color = vignette_color;
  _stale = true;
}

////////////////////////////////////////////////////////////////////
//     Function: ProjectionScreen::get_vignette_color
//       Access: Published
//  Description: Returns the color the screen will be painted at the
//               portions outside of the lens' frustum.  See
//               set_vignette_color().
////////////////////////////////////////////////////////////////////
INLINE const Colorf &ProjectionScreen::
get_vignette_color() const {
  return _vignette_color;
}

////////////////////////////////////////////////////////////////////
//     Function: ProjectionScreen::set_frame_color
//       Access: Published
//  Description: Specifies the color the screen will be painted at the
//               portions outside of the lens' frustum; i.e. where the
//               lens can't see it or illuminate it.  This color is
//               only used if the vignette_on flag is true; see
//               set_vignette_on().
////////////////////////////////////////////////////////////////////
INLINE void ProjectionScreen::
set_frame_color(const Colorf &frame_color) {
  _frame_color = frame_color;
  _stale = true;
}

////////////////////////////////////////////////////////////////////
//     Function: ProjectionScreen::get_frame_color
//       Access: Published
//  Description: Returns the color the screen will be painted at the
//               portions outside of the lens' frustum.  See
//               set_frame_color().
////////////////////////////////////////////////////////////////////
INLINE const Colorf &ProjectionScreen::
get_frame_color() const {
  return _frame_color;
}

////////////////////////////////////////////////////////////////////
//     Function: ProjectionScreen::get_last_screen
//       Access: Public
//  Description: Returns an UpdateSeq corresponding to the last time a
//               screen mesh was generated for the ProjectionScreen.
//               Each time generate_screen() is called, this number is
//               incremented; this allows other objects (like
//               NonlinearImager) to know when they need to recompute
//               themselves.
////////////////////////////////////////////////////////////////////
INLINE const UpdateSeq &ProjectionScreen::
get_last_screen() const {
  return _last_screen;
}
