// Filename: eggNurbsSurface.I
// Created by:  drose (15Feb00)
//
////////////////////////////////////////////////////////////////////
//
// PANDA 3D SOFTWARE
// Copyright (c) 2001 - 2004, Disney Enterprises, Inc.  All rights reserved
//
// All use of this software is subject to the terms of the Panda 3d
// Software license.  You should have received a copy of this license
// along with this source code; you will also find a current copy of
// the license at http://etc.cmu.edu/panda3d/docs/license/ .
//
// To contact the maintainers of this program write to
// panda3d-general@lists.sourceforge.net .
//
////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////
//     Function: EggNurbsSurface::Constructor
//       Access: Public
//  Description:
////////////////////////////////////////////////////////////////////
INLINE EggNurbsSurface::
EggNurbsSurface(const string &name) : EggSurface(name) {
  _u_order = 0;
  _v_order = 0;
}

////////////////////////////////////////////////////////////////////
//     Function: EggNurbsSurface::Copy constructor
//       Access: Public
//  Description:
////////////////////////////////////////////////////////////////////
INLINE EggNurbsSurface::
EggNurbsSurface(const EggNurbsSurface &copy) :
  EggSurface(copy),
  _u_knots(copy._u_knots),
  _v_knots(copy._v_knots),
  _u_order(copy._u_order),
  _v_order(copy._v_order)
{
}

////////////////////////////////////////////////////////////////////
//     Function: EggNurbsSurface::Copy assignment operator
//       Access: Public
//  Description:
////////////////////////////////////////////////////////////////////
INLINE EggNurbsSurface &EggNurbsSurface::
operator = (const EggNurbsSurface &copy) {
  EggSurface::operator = (copy);
  _u_knots = copy._u_knots;
  _v_knots = copy._v_knots;
  _u_order = copy._u_order;
  _v_order = copy._v_order;
  return *this;
}

////////////////////////////////////////////////////////////////////
//     Function: EggNurbsSurface::set_u_order
//       Access: Public
//  Description: Directly changes the order in the U direction to the
//               indicated value (which must be an integer in the
//               range 1 <= u_order <= 4).  If possible, it is
//               preferable to use the setup() method instead of this
//               method, since changing the order directly may result
//               in an invalid surface.
////////////////////////////////////////////////////////////////////
INLINE void EggNurbsSurface::
set_u_order(int u_order) {
  nassertv(u_order >= 1 && u_order <= 4);
  _u_order = u_order;
}

////////////////////////////////////////////////////////////////////
//     Function: EggNurbsSurface::set_v_order
//       Access: Public
//  Description: Directly changes the order in the V direction to the
//               indicated value (which must be an integer in the
//               range 1 <= v_order <= 4).  If possible, it is
//               preferable to use the setup() method instead of this
//               method, since changing the order directly may result
//               in an invalid surface.
////////////////////////////////////////////////////////////////////
INLINE void EggNurbsSurface::
set_v_order(int v_order) {
  nassertv(v_order >= 1 && v_order <= 4);
  _v_order = v_order;
}

////////////////////////////////////////////////////////////////////
//     Function: EggNurbsSurface::set_u_knot
//       Access: Public
//  Description: Resets the value of the indicated knot as indicated.
//               k must be in the range 0 <= k < get_num_u_knots(),
//               and the value must be in the range get_u_knot(k - 1)
//               <= value <= get_u_knot(k + 1).
////////////////////////////////////////////////////////////////////
INLINE void EggNurbsSurface::
set_u_knot(int k, double value) {
  nassertv(k >= 0 && k < (int)_u_knots.size());
  _u_knots[k] = value;
}

////////////////////////////////////////////////////////////////////
//     Function: EggNurbsSurface::set_v_knot
//       Access: Public
//  Description: Resets the value of the indicated knot as indicated.
//               k must be in the range 0 <= k < get_num_v_knots(),
//               and the value must be in the range get_v_knot(k - 1)
//               <= value <= get_v_knot(k + 1).
////////////////////////////////////////////////////////////////////
INLINE void EggNurbsSurface::
set_v_knot(int k, double value) {
  nassertv(k >= 0 && k < (int)_v_knots.size());
  _v_knots[k] = value;
}

////////////////////////////////////////////////////////////////////
//     Function: EggNurbsSurface::set_cv
//       Access: Public
//  Description: Redefines the control vertex associated with a
//               particular u, v coordinate pair.  This is just a
//               shorthand to access the EggPrimitive's normal vertex
//               assignment for a 2-d control vertex.
////////////////////////////////////////////////////////////////////
INLINE void EggNurbsSurface::
set_cv(int ui, int vi, EggVertex *vertex) {
  int vertex_index = get_vertex_index(ui, vi);
  set_vertex(vertex_index, vertex);
}

////////////////////////////////////////////////////////////////////
//     Function: EggNurbsSurface::get_u_order
//       Access: Public
//  Description: Returns the order of the surface in the U direction.
//               The order is the degree of the NURBS equation plus 1;
//               for a typical NURBS, the order is 4.  With this
//               implementation of NURBS, the order must be in the
//               range [1, 4].
////////////////////////////////////////////////////////////////////
INLINE int EggNurbsSurface::
get_u_order() const {
  return _u_order;
}

////////////////////////////////////////////////////////////////////
//     Function: EggNurbsSurface::get_v_order
//       Access: Public
//  Description: Returns the order of the surface in the V direction.
//               The order is the degree of the NURBS equation plus 1;
//               for a typical NURBS, the order is 4.  With this
//               implementation of NURBS, the order must be in the
//               range [1, 4].
////////////////////////////////////////////////////////////////////
INLINE int EggNurbsSurface::
get_v_order() const {
  return _v_order;
}

////////////////////////////////////////////////////////////////////
//     Function: EggNurbsSurface::get_u_degree
//       Access: Public
//  Description: Returns the degree of the surface in the U direction.
//               For a typical NURBS, the degree is 3.
////////////////////////////////////////////////////////////////////
INLINE int EggNurbsSurface::
get_u_degree() const {
  return _u_order - 1;
}

////////////////////////////////////////////////////////////////////
//     Function: EggNurbsSurface::get_v_degree
//       Access: Public
//  Description: Returns the degree of the surface in the V direction.
//               for a typical NURBS, the degree is 3.
////////////////////////////////////////////////////////////////////
INLINE int EggNurbsSurface::
get_v_degree() const {
  return _v_order - 1;
}

////////////////////////////////////////////////////////////////////
//     Function: EggNurbsSurface::get_num_u_knots
//       Access: Public
//  Description: Returns the number of knots in the U direction.
////////////////////////////////////////////////////////////////////
INLINE int EggNurbsSurface::
get_num_u_knots() const {
  return _u_knots.size();
}

////////////////////////////////////////////////////////////////////
//     Function: EggNurbsSurface::get_num_v_knots
//       Access: Public
//  Description: Returns the number of knots in the V direction.
////////////////////////////////////////////////////////////////////
INLINE int EggNurbsSurface::
get_num_v_knots() const {
  return _v_knots.size();
}

////////////////////////////////////////////////////////////////////
//     Function: EggNurbsSurface::get_num_u_cvs
//       Access: Public
//  Description: Returns the number of control vertices that should be
//               present in the U direction.  This is determined by
//               the number of knots and the order; it does not
//               necessarily reflect the number of vertices that have
//               actually been added to the surface.  (However, if the
//               number of vertices in the surface are wrong, the
//               surface is invalid.)
////////////////////////////////////////////////////////////////////
INLINE int EggNurbsSurface::
get_num_u_cvs() const {
  return get_num_u_knots() - get_u_order();
}

////////////////////////////////////////////////////////////////////
//     Function: EggNurbsSurface::get_num_v_cvs
//       Access: Public
//  Description: Returns the number of control vertices that should be
//               present in the V direction.  This is determined by
//               the number of knots and the order; it does not
//               necessarily reflect the number of vertices that have
//               actually been added to the surface.  (However, if the
//               number of vertices in the surface are wrong, the
//               surface is invalid.)
////////////////////////////////////////////////////////////////////
INLINE int EggNurbsSurface::
get_num_v_cvs() const {
  return get_num_v_knots() - get_v_order();
}

////////////////////////////////////////////////////////////////////
//     Function: EggNurbsSurface::get_num_cvs
//       Access: Public
//  Description: Returns the total number of control vertices that
//               *should* be defined for the surface.  This is
//               determined by the number of knots and the order, in
//               each direction; it does not necessarily reflect the
//               number of vertices that have actually been added to
//               the surface.  (However, if the number of vertices in
//               the surface are wrong, the surface is invalid.)
////////////////////////////////////////////////////////////////////
INLINE int EggNurbsSurface::
get_num_cvs() const {
  return get_num_u_cvs() * get_num_v_cvs();
}

////////////////////////////////////////////////////////////////////
//     Function: EggNurbsSurface::get_u_index
//       Access: Public
//  Description: Returns the U index number of the given vertex within
//               the EggPrimitive's linear list of vertices.  An
//               EggNurbsSurface maps a linear list of vertices to its
//               2-d mesh; this returns the U index number that
//               corresponds to the nth vertex in the list.
////////////////////////////////////////////////////////////////////
INLINE int EggNurbsSurface::
get_u_index(int vertex_index) const {
  nassertr(vertex_index >= 0 && vertex_index < get_num_cvs(), 0);
  return vertex_index % get_num_u_cvs();
}

////////////////////////////////////////////////////////////////////
//     Function: EggNurbsSurface::get_v_index
//       Access: Public
//  Description: Returns the V index number of the given vertex within
//               the EggPrimitive's linear list of vertices.  An
//               EggNurbsSurface maps a linear list of vertices to its
//               2-d mesh; this returns the V index number that
//               corresponds to the nth vertex in the list.
////////////////////////////////////////////////////////////////////
INLINE int EggNurbsSurface::
get_v_index(int vertex_index) const {
  nassertr(vertex_index >= 0 && vertex_index < get_num_cvs(), 0);
  return vertex_index / get_num_u_cvs();
}

////////////////////////////////////////////////////////////////////
//     Function: EggNurbsSurface::get_vertex_index
//       Access: Public
//  Description: Returns the index number within the EggPrimitive's
//               list of the control vertex at position ui, vi.
////////////////////////////////////////////////////////////////////
INLINE int EggNurbsSurface::
get_vertex_index(int ui, int vi) const {
  nassertr(ui >= 0 && ui < get_num_u_cvs(), 0);
  nassertr(vi >= 0 && vi < get_num_v_cvs(), 0);
  return vi * get_num_u_cvs() + ui;
}

////////////////////////////////////////////////////////////////////
//     Function: EggNurbsSurface::get_u_knot
//       Access: Public
//  Description: Returns the nth knot value defined in the U
//               direction.
////////////////////////////////////////////////////////////////////
INLINE double EggNurbsSurface::
get_u_knot(int k) const {
  nassertr(k >= 0 && k < (int)_u_knots.size(), 0.0);
  return _u_knots[k];
}

////////////////////////////////////////////////////////////////////
//     Function: EggNurbsSurface::get_v_knot
//       Access: Public
//  Description: Returns the nth knot value defined in the V
//               direction.
////////////////////////////////////////////////////////////////////
INLINE double EggNurbsSurface::
get_v_knot(int k) const {
  nassertr(k >= 0 && k < (int)_v_knots.size(), 0.0);
  return _v_knots[k];
}

////////////////////////////////////////////////////////////////////
//     Function: EggNurbsSurface::get_cv
//       Access: Public
//  Description: Returns the control vertex at the indicate U, V
//               position.
////////////////////////////////////////////////////////////////////
INLINE EggVertex *EggNurbsSurface::
get_cv(int ui, int vi) const {
  int vertex_index = get_vertex_index(ui, vi);
  return get_vertex(vertex_index);
}
