// Filename: imageWriter.cxx
// Created by:  drose (19Jun00)
//
////////////////////////////////////////////////////////////////////
//
// PANDA 3D SOFTWARE
// Copyright (c) 2001 - 2004, Disney Enterprises, Inc.  All rights reserved
//
// All use of this software is subject to the terms of the Panda 3d
// Software license.  You should have received a copy of this license
// along with this source code; you will also find a current copy of
// the license at http://etc.cmu.edu/panda3d/docs/license/ .
//
// To contact the maintainers of this program write to
// panda3d-general@lists.sourceforge.net .
//
////////////////////////////////////////////////////////////////////

#include "imageWriter.h"

////////////////////////////////////////////////////////////////////
//     Function: ImageWriter::Constructor
//       Access: Public
//  Description: Image-writing type programs *must* specify their
//               output file using -o.
////////////////////////////////////////////////////////////////////
ImageWriter::
ImageWriter(bool allow_last_param) :
  WithOutputFile(allow_last_param, false, true)
{
  clear_runlines();
  if (_allow_last_param) {
    add_runline("[opts] outputimage");
  }
  add_runline("[opts] -o outputimage");

  string o_description;
  if (_allow_last_param) {
    o_description =
      "Specify the filename to which the resulting image file will be written.  "
      "If this option is omitted, the last parameter name is taken to be the "
      "name of the output file.";
  } else {
    o_description =
      "Specify the filename to which the resulting image file will be written.";
  }

  add_option
    ("o", "filename", 50, o_description,
     &ImageWriter::dispatch_filename, &_got_output_filename, &_output_filename);
}


////////////////////////////////////////////////////////////////////
//     Function: ImageWriter::write_image
//       Access: Public
//  Description: Writes the generated to the user's specified output
//               filename.
////////////////////////////////////////////////////////////////////
void ImageWriter::
write_image(const PNMImage &image) {
  if (!image.write(get_output_filename())) {
    nout << "Unable to write output image to " 
         << get_output_filename() << "\n";
    exit(1);
  }
}

////////////////////////////////////////////////////////////////////
//     Function: ImageWriter::handle_args
//       Access: Protected, Virtual
//  Description: Does something with the additional arguments on the
//               command line (after all the -options have been
//               parsed).  Returns true if the arguments are good,
//               false otherwise.
////////////////////////////////////////////////////////////////////
bool ImageWriter::
handle_args(ProgramBase::Args &args) {
  if (!check_last_arg(args, 0)) {
    return false;
  }

  if (!args.empty()) {
    nout << "Unexpected arguments on command line:\n";
    Args::const_iterator ai;
    for (ai = args.begin(); ai != args.end(); ++ai) {
      nout << (*ai) << " ";
    }
    nout << "\r";
    return false;
  }

  return true;
}
