/**
 * PANDA 3D SOFTWARE
 * Copyright (c) Carnegie Mellon University.  All rights reserved.
 *
 * All use of this software is subject to the terms of the revised BSD
 * license.  You should have received a copy of this license along
 * with this source code in a file named "LICENSE."
 *
 * @file pnmTextMaker.I
 * @author drose
 * @date 2003-09-07
 */

/**
 * Returns true if the PNMTextMaker is valid and ready to generate text, false
 * otherwise.
 */
INLINE bool PNMTextMaker::
is_valid() const {
  return _is_valid;
}

/**
 *
 */
INLINE void PNMTextMaker::
set_align(PNMTextMaker::Alignment align_type) {
  _align = align_type;
}

/**
 *
 */
INLINE PNMTextMaker::Alignment PNMTextMaker::
get_align() const {
  return _align;
}

/**
 * Sets the flag that indicates whether the interior of hollow fonts is
 * identified as a preprocess as each glyph is loaded.  If this flag is true,
 * you may specify an interior color along with a fg and bg color when you
 * place text; if the flag is false, the interior color is ignored.
 *
 * It is generally best to set_native_antialias(0) when using this feature.
 * Also, this works best when the pixel size is not very small.
 */
INLINE void PNMTextMaker::
set_interior_flag(bool interior_flag) {
  if (_interior_flag != interior_flag) {
    _interior_flag = interior_flag;
    empty_cache();
  }
}

/**
 *
 */
INLINE bool PNMTextMaker::
get_interior_flag() const {
  return _interior_flag;
}

/**
 * Sets the foreground color of text that will be generated by future calls to
 * generate_into().  This is the color that all of the "on" pixels in the font
 * will show as.
 */
INLINE void PNMTextMaker::
set_fg(const LColor &fg) {
  _fg = fg;
}

/**
 * Returns the foreground color of text that will be generated by future calls
 * to generate_into().
 */
INLINE const LColor &PNMTextMaker::
get_fg() const {
  return _fg;
}

/**
 * Sets the color that will be used to render the interior portions of hollow
 * fonts in future calls to generate_into().  This is respected only if
 * interior_flag is true.
 */
INLINE void PNMTextMaker::
set_interior(const LColor &interior) {
  _interior = interior;
}

/**
 * Returns the color that will be used to render the interior portions of
 * hollow fonts.
 */
INLINE const LColor &PNMTextMaker::
get_interior() const {
  return _interior;
}

/**
 * If this is set to something other than 0, Panda will generate a signed
 * distance field with the given radius.
 */
INLINE void PNMTextMaker::
set_distance_field_radius(int distance_field_radius) {
  _distance_field_radius = distance_field_radius;
}

/**
 * Returns the radius previously set with set_distance_field_radius, or 0
 * otherwise.
 */
INLINE int PNMTextMaker::
get_distance_field_radius() const {
  return _distance_field_radius;
}

/**
 * Generates a single line of text into the indicated image at the indicated
 * position; the return value is the total width in pixels.
 */
INLINE int PNMTextMaker::
generate_into(const std::string &text, PNMImage &dest_image, int x, int y) {
  TextEncoder encoder;
  encoder.set_text(text);
  return generate_into(encoder.get_wtext(), dest_image, x, y);
}

/**
 * Returns the width in pixels of the indicated line of text.
 */
INLINE int PNMTextMaker::
calc_width(const std::string &text) {
  TextEncoder encoder;
  encoder.set_text(text);
  return calc_width(encoder.get_wtext());
}
