/**
 * PANDA 3D SOFTWARE
 * Copyright (c) Carnegie Mellon University.  All rights reserved.
 *
 * All use of this software is subject to the terms of the revised BSD
 * license.  You should have received a copy of this license along
 * with this source code in a file named "LICENSE."
 *
 * @file spotlight.I
 * @author mike
 * @date 1999-02-04
 */

/**
 *
 */
INLINE Spotlight::CData::
CData() :
  _exponent(50.0f),
  _specular_color(1.0f, 1.0f, 1.0f, 1.0f),
  _attenuation(1.0f, 0.0f, 0.0f),
  _max_distance(make_inf((PN_stdfloat)0))
{
}

/**
 *
 */
INLINE Spotlight::CData::
CData(const Spotlight::CData &copy) :
  _exponent(copy._exponent),
  _specular_color(copy._specular_color),
  _attenuation(copy._attenuation),
  _max_distance(copy._max_distance)
{
}

/**
 * Returns the exponent that controls the amount of light falloff from the
 * center of the spotlight.  See set_exponent().
 */
INLINE PN_stdfloat Spotlight::
get_exponent() const {
  CDReader cdata(_cycler);
  return cdata->_exponent;
}

/**
 * Sets the exponent that controls the amount of light falloff from the center
 * of the spotlight.  The light is attenuated by the cosine of the angle
 * between the direction of the light and the direction of the point being
 * lighted, raised to the power of this exponent.  Thus, higher exponents
 * result in a more focused light source, regardless of the field-of-view of
 * the lens.
 */
INLINE void Spotlight::
set_exponent(PN_stdfloat exponent) {
  CDWriter cdata(_cycler);
  cdata->_exponent = exponent;
}

/**
 * Returns the color of specular highlights generated by the light.  This is
 * usually the same as get_color().
 */
INLINE const LColor &Spotlight::
get_specular_color() const {
  if (_has_specular_color) {
    CDReader cdata(_cycler);
    return cdata->_specular_color;
  } else {
    return get_color();
  }
}

/**
 * Sets the color of specular highlights generated by the light.
 */
INLINE void Spotlight::
set_specular_color(const LColor &color) {
  CDWriter cdata(_cycler);
  _has_specular_color = true;
  cdata->_specular_color = color;
}

/**
 * Clears a custom specular color setting, meaning that the specular color
 * will now come from the color.
 */
INLINE void Spotlight::
clear_specular_color() {
  _has_specular_color = false;
}

/**
 * Returns the terms of the attenuation equation for the light.  These are, in
 * order, the constant, linear, and quadratic terms based on the distance from
 * the point to the vertex.
 */
INLINE const LVecBase3 &Spotlight::
get_attenuation() const {
  CDReader cdata(_cycler);
  return cdata->_attenuation;
}

/**
 * Sets the terms of the attenuation equation for the light.  These are, in
 * order, the constant, linear, and quadratic terms based on the distance from
 * the point to the vertex.
 */
INLINE void Spotlight::
set_attenuation(const LVecBase3 &attenuation) {
  CDWriter cdata(_cycler);
  cdata->_attenuation = attenuation;
}

/**
 * Returns the maximum distance at which the light has any effect, as previously
 * specified by set_max_distance.
 */
INLINE PN_stdfloat Spotlight::
get_max_distance() const {
  CDReader cdata(_cycler);
  return cdata->_max_distance;
}

/**
 * Sets the radius of the light's sphere of influence.  Beyond this distance, the
 * light may be attenuated to zero, if this is supported by the shader.
 */
INLINE void Spotlight::
set_max_distance(PN_stdfloat max_distance) {
  CDWriter cdata(_cycler);
  cdata->_max_distance = max_distance;
}
