/**
 * PANDA 3D SOFTWARE
 * Copyright (c) Carnegie Mellon University.  All rights reserved.
 *
 * All use of this software is subject to the terms of the revised BSD
 * license.  You should have received a copy of this license along
 * with this source code in a file named "LICENSE."
 *
 * @file dcField.I
 * @author drose
 * @date 2006-01-10
 */

/**
 * Returns a unique index number associated with this field.  This is defined
 * implicitly when the .dc file(s) are read.
 */
INLINE int DCField::
get_number() const {
  return _number;
}

/**
 * Returns the DCClass pointer for the class that contains this field.
 */
INLINE DCClass *DCField::
get_class() const {
  return _dclass;
}

/**
 * Returns true if a default value has been explicitly established for this
 * field, false otherwise.
 */
INLINE bool DCField::
has_default_value() const {
  return _has_default_value;
}

/**
 * Returns the default value for this field.  If a default value has been
 * explicitly set (e.g.  has_default_value() returns true), returns that
 * value; otherwise, returns an implicit default for the field.
 */
INLINE const vector_uchar &DCField::
get_default_value() const {
  if (_default_value_stale) {
    ((DCField *)this)->refresh_default_value();
  }
  return _default_value;
}

/**
 * Returns true if the field has been flagged as a bogus field.  This is set
 * for fields that are generated by the parser as placeholder for missing
 * fields, as when reading a partial file; it should not occur in a normal
 * valid dc file.
 */
INLINE bool DCField::
is_bogus_field() const {
  return _bogus_field;
}

/**
 * Returns true if the "required" flag is set for this field, false otherwise.
 */
INLINE bool DCField::
is_required() const {
  return has_keyword("required");
}

/**
 * Returns true if the "broadcast" flag is set for this field, false
 * otherwise.
 */
INLINE bool DCField::
is_broadcast() const {
  return has_keyword("broadcast");
}

/**
 * Returns true if the "ram" flag is set for this field, false otherwise.
 */
INLINE bool DCField::
is_ram() const {
  return has_keyword("ram");
}

/**
 * Returns true if the "db" flag is set for this field, false otherwise.
 */
INLINE bool DCField::
is_db() const {
  return has_keyword("db");
}

/**
 * Returns true if the "clsend" flag is set for this field, false otherwise.
 */
INLINE bool DCField::
is_clsend() const {
  return has_keyword("clsend");
}

/**
 * Returns true if the "clrecv" flag is set for this field, false otherwise.
 */
INLINE bool DCField::
is_clrecv() const {
  return has_keyword("clrecv");
}

/**
 * Returns true if the "ownsend" flag is set for this field, false otherwise.
 */
INLINE bool DCField::
is_ownsend() const {
  return has_keyword("ownsend");
}

/**
 * Returns true if the "ownrecv" flag is set for this field, false otherwise.
 */
INLINE bool DCField::
is_ownrecv() const {
  return has_keyword("ownrecv");
}

/**
 * Returns true if the "airecv" flag is set for this field, false otherwise.
 */
INLINE bool DCField::
is_airecv() const {
  return has_keyword("airecv");
}

/**
 * Write a string representation of this instance to <out>.
 */
INLINE void DCField::
output(std::ostream &out) const {
  output(out, true);
}

/**
 * Write a string representation of this instance to <out>.
 */
INLINE void DCField::
write(std::ostream &out, int indent_level) const {
  write(out, false, indent_level);
}

/**
 * Assigns the unique number to this field.  This is normally called only by
 * the DCClass interface as the field is added.
 */
INLINE void DCField::
set_number(int number) {
  _number = number;
}

/**
 * Assigns the class pointer to this field.  This is normally called only by
 * the DCClass interface as the field is added.
 */
INLINE void DCField::
set_class(DCClass *dclass) {
  _dclass = dclass;
}

/**
 * Establishes a default value for this field.
 */
INLINE void DCField::
set_default_value(vector_uchar default_value) {
  _default_value = std::move(default_value);
  _has_default_value = true;
  _default_value_stale = false;
}
