/**
 * PANDA 3D SOFTWARE
 * Copyright (c) Carnegie Mellon University.  All rights reserved.
 *
 * All use of this software is subject to the terms of the revised BSD
 * license.  You should have received a copy of this license along
 * with this source code in a file named "LICENSE."
 *
 * @file prcKeyRegistry.h
 * @author drose
 * @date 2004-10-19
 */

#ifndef PRCKEYREGISTRY_H
#define PRCKEYREGISTRY_H

#include "dtoolbase.h"

// This file requires OpenSSL to compile, because we use routines in the
// OpenSSL library to manage keys and to sign and validate signatures.

#ifdef HAVE_OPENSSL

#include <vector>

typedef struct evp_pkey_st EVP_PKEY;

/**
 * This class records the set of public keys used to verify the signature on a
 * prc file.  The actual public keys themselves are generated by the make-prc-
 * key utility; the output of this utility is a .cxx file which should be
 * named by the PRC_PUBLIC_KEYS_FILENAME variable in Config.pp.
 *
 * This class requires the OpenSSL library.
 */
class EXPCL_DTOOL_PRC PrcKeyRegistry {
protected:
  PrcKeyRegistry();
  ~PrcKeyRegistry();

public:
  struct KeyDef {
    const char *_data;
    size_t _length;
    time_t _generated_time;
  };

  void record_keys(const KeyDef *key_def, size_t num_keys);
  void set_key(size_t n, EVP_PKEY *pkey, time_t generated_time);

  size_t get_num_keys() const;
  EVP_PKEY *get_key(size_t n) const;
  time_t get_generated_time(size_t n) const;

  static PrcKeyRegistry *get_global_ptr();

private:

  class Key {
  public:
    const KeyDef *_def;
    EVP_PKEY *_pkey;
    time_t _generated_time;
  };

  typedef std::vector<Key> Keys;
  Keys _keys;

  static PrcKeyRegistry *_global_ptr;
};

#include "prcKeyRegistry.I"

#endif  // HAVE_OPENSSL

#endif
